define(function (require, exports, module) {
    'use strict';
    var PREFIX = 'bb.beautify',
        PREF_DIALOG_ID = 'hideDialog',
        DocumentManager = brackets.getModule('document/DocumentManager'),
        Editor = brackets.getModule('editor/Editor').Editor,
        EditorManager = brackets.getModule('editor/EditorManager'),
        LiveDevelopment,
        PreferencesManager = brackets.getModule('preferences/PreferencesManager'),
        Mustache = brackets.getModule('thirdparty/mustache/mustache'),
        DefaultDialogs = brackets.getModule('widgets/DefaultDialogs'),
        Dialogs = brackets.getModule('widgets/Dialogs'),
        Strings = require('strings'),
        DialogContentTemplate = require('text!templates/dialog-content.html'),
        DialogContent = Mustache.render(DialogContentTemplate, {
            Strings: Strings
        }),
        prefs = PreferencesManager.getExtensionPrefs(PREFIX),
        beautifyWeb = {
            js: require('thirdparty/beautify').js_beautify,
            css: require('thirdparty/beautify-css').css_beautify,
            html: require('thirdparty/beautify-html').html_beautify,
            php: require('thirdparty/beautify-php').php_beautify
        };
    /**
     * Check if LiveDevelopment is currently active.
     * The LiveDevelopment module has to be requested conditionally as Brackets-Electron doesn't support it.
     * See https://github.com/zaggino/brackets-electron/issues/25
     * @returns {boolean} True, if LiveDevelopment is currently active.
     */
    // Cloned from main.js
    function isLiveDevelopmentActive() {
        if (window.electron) {
            return false;
        } else {
            if (!LiveDevelopment) {
                LiveDevelopment = brackets.getModule('LiveDevelopment/LiveDevelopment');
            }
            return LiveDevelopment.status === LiveDevelopment.STATUS_ACTIVE;
        }
    }

    // Moved this code out of format(autoSave) function, so that this can be configured from outside.
    function checkBeforeReplacing(unformattedText, formattedText, range, batchUpdateProc) {
        if (formattedText !== unformattedText) {
            batchUpdateProc(formattedText, range);
        }
    }

    /**
    /* Returns the indent size based on DW indent preferences.
    */
   function getIndentSize() {
        var indentSize = Editor.getDWPrefForIndentSize();
        if (Editor.getDWPrefForUseTabsInIndent()) {
            if (!Editor.getUseTabChar()) {
                indentSize *= Editor.getTabSize();
            }
        }
        return indentSize;
    }

    /**
    /* Returns whether to use spaces or tabs in indent based on DW preferences.
    */
    function useSpacesInIndent() {
        if (Editor.getDWPrefForUseTabsInIndent()) {
            return !Editor.getUseTabChar();
        } else {
            return true;
        }
    }

    // Setting preferences based on DW prefs.
    function setIndentPrefs(currentOptions) {
        currentOptions.indent_size = getIndentSize();
        if (useSpacesInIndent()) {
            currentOptions.indent_char = ' ';
        } else {
            currentOptions.indent_with_tabs = true;
            var indent_char = '\t';
            var indentString = indent_char.repeat(currentOptions.indent_size);
            currentOptions.indent_char = indentString;
        }
        return currentOptions;
    }

    // [DW-START] Cloning Original format function to cater to DW specific formatting requirements
    // While integrating new version of BracketsBeautify, format function should be cloned carefully to cater all DW use cases
    // DW has following use cases:
    // formatDWFile => Format entire file , ignore the selection, as DW Command format source has been invoked
    // formatDWText => Format text array sent from DW
    // formatDWEmbeddedCode => format code snippet with the laguage specified by DW
    function formatDWInternal(beautifier, documentHasSelection, documentSelectedText, batchUpdateProc, shouldIncludeIndentChars, options, autoSave) {
        var document = DocumentManager.getCurrentDocument();

        if (!beautifier && !autoSave) {
            var Dialog = Dialogs.showModalDialog(DefaultDialogs.DIALOG_ID_ERROR, Strings.UNSUPPORTED_TITLE, DialogContent);
            Dialog.getPromise().done(function () {
                prefs.set(PREF_DIALOG_ID, Dialog.getElement().find('input').prop('checked'));
            });
            return;
        }

        var unformattedText;
        var editor = EditorManager.getCurrentFullEditor();
        var currentOptions = setIndentPrefs($.extend({}, options[beautifier] || options));

        var range;
        if (documentHasSelection) {
            currentOptions.end_with_newline = false;
            unformattedText = documentSelectedText;
            if (shouldIncludeIndentChars) {
                range = editor.getSelection();

                /*
                * When we are only formatting a range, we still want to have it correctly indented with the flow.
                * The library has an option for that (indent_level), however that doesn't seem to work.
                 * See open issue: https://github.com/beautify-web/js-beautify/issues/724).
                 * As a temporary solution we are checking if the starting line of the selection has some unselected
                * indentation and if so extending the selection.
                */
                // currentOptions.indent_level = range.start.ch;
                var indentChars = document.getLine(range.start.line).substr(0, range.start.ch);
                if (indentChars.trim().length === 0) {
                    range.start.ch = 0;
                    unformattedText = indentChars + unformattedText;
                }
            }
        } else {
            unformattedText = document.getText();
            /*
             * If the current document is html and is currently used in LiveDevelopment, we must not change the html tag
             * as that causes the DOM in the browser to duplicate (see https://github.com/adobe/brackets/issues/10634).
             * To prevent that, we select the content inside <html> if we can find one and pretend a selection for the
             * formatting and replacing.
             * NOTE: Currently it is only checked if LiveDevelopment is active in general as I don't know how to check
             * for a specific file (see https://groups.google.com/forum/#!topic/brackets-dev/9wEtqG684cI).
             */
            if (document.getLanguage().getId() === 'html' && isLiveDevelopmentActive()) {
                // Regex to match everything inside <html> beginning by the first tag and ending at the last
                var match = /((?:.|\n)*?<html[^>]*?>\s*)((?:.|\n)*)(\s*<\/html>)/gm.exec(unformattedText);
                if (match) {
                    unformattedText = match[2];
                    range = {
                        start: {
                            line: match[1].split('\n').length - 1,
                            ch: match[1].length - match[1].lastIndexOf('\n') - 1
                        },
                        end: {
                            line: (match[1] + match[2]).split('\n').length - 1,
                            ch: (match[1] + match[2]).length - (match[1] + match[2]).lastIndexOf('\n') - 1
                        }
                    };
                    currentOptions.end_with_newline = false;
                }
            }
        }

        if (beautifyWeb[beautifier]) {
            var formattedText = beautifyWeb[beautifier](unformattedText, currentOptions);
            checkBeforeReplacing(unformattedText, formattedText, range, batchUpdateProc);
        }
    }

    function alignWithFormatted(formattedSnapshot, unformattedText, unformattedTextBegIdx, unformattedTextEndIdx, formattedSnapshotBegIdx) {
        if (unformattedTextBegIdx < 0 || unformattedTextEndIdx < 0 || formattedSnapshotBegIdx < 0) {
            return -1;
        }
        var idx1 = unformattedTextBegIdx, idx2 = formattedSnapshotBegIdx, whitespaceRegex = /\s/;
        while (idx1 < unformattedTextEndIdx && idx2 < formattedSnapshot.length) {
            if (unformattedText[idx1] === formattedSnapshot[idx2]) {
                idx1 += 1;
                idx2 += 1;
            } else if (whitespaceRegex.test(unformattedText[idx1])) {
                idx1 += 1;
            } else if (whitespaceRegex.test(formattedSnapshot[idx2])) {
                idx2 += 1;
            } else {
                return -1;
            }
        }
        if (idx1 < unformattedTextEndIdx) {
            return -1;
        }
        return idx2;
    }

    // Fixing special case of Dreamweaver corrupting a code snippet like </h<?php echo $header_level ?>>
    // We know for sure that PHP tag will not be inside double quotes, i.e. an attribute value
    function gotoMachingQuote(text, idx, skip) {
        while (idx >= 0 && idx < text.length && text[idx] !== '"') {
            idx += 1;
        }
        return idx;
    }
    function gotoEnclosingTag(text, idx, backwards) {
        backwards = backwards || false;
        var skip = backwards ? -1 : 1;
        while (idx >= 0 && idx < text.length && text[idx] !== '<' && text[idx] !== '>') {
            if (text[idx] === '"') {
                idx = gotoMachingQuote(text, idx + skip, skip);
            }
            idx += skip;
        }
        return idx;
    }

    function getEnclosingHtmlTag(text, begPhp, endPhp) {
        if (begPhp > 0 && endPhp < text.length) {
            var enclosingTagLeft = gotoEnclosingTag(text, begPhp - 1, true),
                enclosingTagRight = gotoEnclosingTag(text, endPhp);
            if (enclosingTagRight < text.length && enclosingTagLeft >= 0
                    && text[enclosingTagLeft] === '<' && text[enclosingTagRight] === '>') {
                return {
                    left: enclosingTagLeft,
                    right: enclosingTagRight + 1
                };
            }
        }
        return {
            left: -1,
            right: -1
        };
    }

    function skipWhitespaces(text, idx, backwards) {
        backwards = backwards || false;
        var skip = backwards ? -1 : 1,
            whitespaceRegex = /\s/;
        while (idx >= 0 && idx < text.length && whitespaceRegex.test(text[idx])) {
            idx += skip;
        }
        return idx;
    }

    function skipWhitespacesBackward(text, idx) {
        return skipWhitespaces(text, idx, true);
    }

    function skipWhitespacesForward(text, idx) {
        return skipWhitespaces(text, idx);
    }

    function checkIfEndsWithLF (txt) {
        var lastNonWhiteCharIdx = skipWhitespacesBackward(txt, txt.length - 1),
            newLineAfterLastNonWhiteChar = txt.indexOf('\n', lastNonWhiteCharIdx),
            retval = {
                actualEnd: lastNonWhiteCharIdx + 1,
                endsWithLF: false
            };
        if (newLineAfterLastNonWhiteChar > lastNonWhiteCharIdx && newLineAfterLastNonWhiteChar < txt.length) {
            retval.endsWithLF = true;
        }
        return retval;
    }

    function formatDelegatedTextAndPaste(formattedSnapshot, codeOffsetArr, batchUpdateProc, beautifyPrefs) {
        if (!Array.isArray(codeOffsetArr) || codeOffsetArr.length < 2) {
            console.debug('\nInsufficient args to formatDelegatedTextAndPaste: ', codeOffsetArr);
            return;
        }
        var selLimits = codeOffsetArr.pop();
        if(!Array.isArray(selLimits) || selLimits[0] !== 'selLimits' || isNaN(selLimits[1]) || isNaN(selLimits[2])) {
            console.debug('\nSelection information was not valid: ', selLimits);
            return;
        }
        var dataSentToFormatter;
        var saveFormatted = function (formattedText, range) {
            if (dataSentToFormatter.text) {
                // We had prefixed input text to formatter with line-indent, so need to discard it
                var actualBegin = skipWhitespacesForward(formattedText, 0);
                formattedSnapshot = formattedSnapshot.substring(0, dataSentToFormatter.beg) + formattedText.substring(actualBegin) + formattedSnapshot.substring(dataSentToFormatter.end);
            }
        };
        var currEditor = EditorManager.getCurrentFullEditor(),
            editRange = {
                start: currEditor.posFromIndex(selLimits[1]),
                end: currEditor.posFromIndex(selLimits[2]),
                reversed: false
            },
            allUnformattedText = currEditor.document.getRange(editRange.start, editRange.end);
        if (!editRange || !allUnformattedText) {
            console.debug('\nNo selection information was found for: ', selLimits);
            return;
        }
        var idx,
            size = codeOffsetArr.length,
            formattedRemaining = formattedSnapshot.length,
            unformattedDone = 0;
        for (idx = 0; idx < size; idx += 1) {
            dataSentToFormatter = {};
            if (!Array.isArray(codeOffsetArr[idx]) || codeOffsetArr[idx].length !== 3) {
                console.debug('\nNot a valid code offset entry: ', codeOffsetArr[idx]);
                continue;
            }
            var endPhpTag = codeOffsetArr[idx][2],
                begPhpTag = codeOffsetArr[idx][1],
                beautifierId = codeOffsetArr[idx][0];
            if (!beautifierId || isNaN(begPhpTag) || isNaN(endPhpTag)) {
                console.debug('\nNot a valid code offset entry: ', codeOffsetArr[idx]);
                continue;
            }
            if(begPhpTag >= endPhpTag || (selLimits[1] + endPhpTag) > selLimits[2]) {
                console.debug('\nNot a valid code offset entry: ', codeOffsetArr[idx], " for selection: ", selLimits);
                var codeRange = {
                    start: currEditor.posFromIndex(selLimits[1] + begPhpTag),
                    end: currEditor.posFromIndex(selLimits[1] + endPhpTag),
                    reversed: false
                };
                console.debug('\nEditRange: ', editRange, " and code range: ", codeRange);
                continue;
            }
            var formattedDone = formattedSnapshot.length - formattedRemaining,
                begPhpInFormatted = alignWithFormatted(formattedSnapshot, allUnformattedText, unformattedDone, begPhpTag, formattedDone);
            begPhpInFormatted = skipWhitespacesForward(formattedSnapshot, begPhpInFormatted);
            if (begPhpInFormatted < 0) {
                console.debug('\nDW formatted was different:\n', formattedSnapshot.substring(formattedDone));
                console.debug('\nCompared to original:\n', allUnformattedText.substring(unformattedDone));
                return;
            }
            var endPhpInFormatted = alignWithFormatted(formattedSnapshot, allUnformattedText, begPhpTag, endPhpTag, begPhpInFormatted);
            if (endPhpInFormatted <= 0) { // Skip whitespaces return index to first non-white char
                console.debug('\nDW formatted was different:\n', formattedSnapshot.substring(begPhpInFormatted));
                console.debug('\nCompared to original:\n', allUnformattedText.substring(begPhpTag));
                return;
            }
            formattedRemaining = formattedSnapshot.length - endPhpInFormatted;
            unformattedDone = endPhpTag;
            var enclosingHtmlTagLimits = getEnclosingHtmlTag(allUnformattedText, begPhpTag, endPhpTag);
            if (enclosingHtmlTagLimits.left >= 0 && enclosingHtmlTagLimits.right > 0) {
                var enclosingHtmlTagLimitsFormatted = getEnclosingHtmlTag(formattedSnapshot, begPhpInFormatted, endPhpInFormatted);
                if (enclosingHtmlTagLimitsFormatted.left >= 0 && enclosingHtmlTagLimitsFormatted.right > 0) {
                    begPhpInFormatted = enclosingHtmlTagLimitsFormatted.left;
                    endPhpInFormatted = enclosingHtmlTagLimitsFormatted.right;
                    begPhpTag = enclosingHtmlTagLimits.left;
                    endPhpTag = enclosingHtmlTagLimits.right;
                    dataSentToFormatter = {
                        beg: begPhpInFormatted,
                        end: endPhpInFormatted,
                        text: allUnformattedText.substring(begPhpTag, endPhpTag)
                    };
                    // Need to update the offsets, as we have considered enclosing HTML as well.
                    formattedRemaining = formattedSnapshot.length - endPhpInFormatted;
                    unformattedDone = endPhpTag;
                    // need to replace formatted with original, as DW has broke the code.
                    // This happens for snippet like </h<?php echo $headerLevel; ?>>
                    // DW will break this snippet like:
                    // </h
                    // <?php echo $headerLevel; ?>
                    // >
                    // Which causes code corruption.
                    saveFormatted(dataSentToFormatter.text);
					// Check if we've another PHP tag to deal with in the enclosing html tag, simply ignore it.
					var next_index = idx + 1;
					while (next_index < size && codeOffsetArr[next_index][1] > begPhpTag && codeOffsetArr[next_index][2] < endPhpTag) {
						idx = next_index;
						next_index = idx + 1;
					}
                } else {
                    console.debug('\nEnclosing html was following:\n',
                        allUnformattedText.substring(enclosingHtmlTagLimitsFormatted.left, enclosingHtmlTagLimitsFormatted.right));
                    console.debug('\nBut same could not be found in DW formatted:\n', formattedSnapshot.substring(0, formattedRemaining));
                    return;
                }
            } else {
                dataSentToFormatter = {
                    beg: begPhpInFormatted,
                    end: endPhpInFormatted,
                    text: allUnformattedText.substring(begPhpTag, endPhpTag)
                };
                var lastNewLine = formattedSnapshot.lastIndexOf('\n', begPhpInFormatted), // if -1 is returned, then it means begin of selection
	               indentChars;
                if (lastNewLine < 0) {
	               indentChars = currEditor.document.getLine(editRange.start.line).substring(0, editRange.start.ch);
	               if (indentChars.trim().length === 0) {
		              // Join indent whitespaces from begining of selection line(before selection) and indent whitespaces in formattedSnapshot
		              indentChars = indentChars + formattedSnapshot.substring(0, skipWhitespacesForward(formattedSnapshot, 0));
	               } else {
		              // Only care for indent whitespaces from begining of selection line
		              // As there are some content between begining of selection and begining of PHP tag.
		              indentChars = indentChars.substring(0, skipWhitespacesForward(indentChars, 0));
	               }
                } else {
	               // Prefix indent (from begining of line to first mon-white char)
	               indentChars = formattedSnapshot.substring(lastNewLine + 1, skipWhitespacesForward(formattedSnapshot, lastNewLine));
                }
                var tobesent = indentChars + formattedSnapshot.substring(begPhpInFormatted, endPhpInFormatted); // prepend indent to PHP
                formatDWInternal(beautifierId, true, tobesent, saveFormatted, false, beautifyPrefs, false);
            }
        }
        checkBeforeReplacing(allUnformattedText, formattedSnapshot, editRange, batchUpdateProc);
    }

    // Format entire file, ignore the selection information
    function formatDWFile(beautifierId, batchUpdateProc, beautifyPrefs, autoSave) {
        formatDWInternal(beautifierId, false, '', batchUpdateProc, false, beautifyPrefs, autoSave);
    }

    // Format selected embedded PHP/CSS/JS code, inside HTML/PHP
    function formatDWEmbeddedCode (args, batchUpdateProc, beautifyPrefs, autoSave) {
        var currEditor = EditorManager.getCurrentFullEditor();
        if (currEditor.hasSelection()) { // No point calling format selection, if no code is selected.
            var errorMessage = function () {
                console.debug('\nCould not process args for formatEmbeddedCode: ', args);
                return;
            };
            var arr = args.split(',');
            if (!Array.isArray(arr) || arr.length !== 2) {
                return errorMessage();
            }
            var beautifierId = arr[0],
                caretBegin = parseInt(arr[1], 10);
            if (!beautifierId || isNaN(caretBegin)) {
                return errorMessage();
            }
            var editRange = currEditor.getSelection(),
                unselectedRange = {
                    start: currEditor.posFromIndex(caretBegin),
                    end: editRange.start,
                    reversed: false
                },
                indentChars = currEditor.document.getLine(unselectedRange.start.line),
                firstNonWhiteCharInIndent = skipWhitespacesForward(indentChars, 0),
                allUnformattedText = currEditor.document.getRange(unselectedRange.start, unselectedRange.end),
                begOffsetSelectedText = firstNonWhiteCharInIndent + allUnformattedText.length,
                actualSelectedText = currEditor.document.getRange(editRange.start, editRange.end),
                actualFormattedText = actualSelectedText;
            allUnformattedText = indentChars.substring(0, firstNonWhiteCharInIndent) + allUnformattedText + actualSelectedText;
            if (!editRange || !allUnformattedText || !unselectedRange) {
                return errorMessage();
            }
            var saveFormatted = function (formattedText, range) {
                var begInFormatted = alignWithFormatted(formattedText, allUnformattedText, 0, begOffsetSelectedText, 0);
                if(begInFormatted >= 0) {
                    actualFormattedText = formattedText.substring(begInFormatted);
                }
            };
            formatDWInternal(beautifierId, true, allUnformattedText, saveFormatted, false, beautifyPrefs, autoSave);
            var newLineInUnformatted = checkIfEndsWithLF(actualSelectedText);
            if (newLineInUnformatted.endsWithLF){
                var newLineInFormatted = checkIfEndsWithLF(actualFormattedText);
                if (!newLineInFormatted.endsWithLF) {
                    actualFormattedText = actualFormattedText.substring(0, newLineInFormatted.actualEnd) + '\n';
                }
            }
            checkBeforeReplacing(actualSelectedText, actualFormattedText, editRange, batchUpdateProc);
        }
    }

    exports.formatDelegatedTextAndPaste = formatDelegatedTextAndPaste;
    exports.formatDWEmbeddedCode = formatDWEmbeddedCode;
    exports.formatDWFile = formatDWFile;
    exports.setIndentPrefs = setIndentPrefs;
    // [DW-END]
});
